const root = document.documentElement;
const themeToggleBtn = document.querySelector("[data-theme-toggle]");
const menuToggleBtn = document.querySelector(".menu-toggle");
const navbar = document.querySelector(".navbar");
const THEME_KEY = "site-theme";

function safeGetTheme() {
  try {
    return localStorage.getItem(THEME_KEY);
  } catch {
    return null;
  }
}

function safeSetTheme(theme) {
  try {
    localStorage.setItem(THEME_KEY, theme);
  } catch {
    // ignore
  }
}

function applyTheme(theme) {
  if (theme !== "light" && theme !== "dark") return;
  root.setAttribute("data-theme", theme);
  safeSetTheme(theme);
}

// Initial theme
const savedTheme = safeGetTheme();

if (savedTheme === "light" || savedTheme === "dark") {
  applyTheme(savedTheme);
} else {
  const prefersLight =
    window.matchMedia &&
    window.matchMedia("(prefers-color-scheme: light)").matches;
  applyTheme(prefersLight ? "light" : "dark");
}

// Theme toggle button
if (themeToggleBtn) {
  themeToggleBtn.addEventListener("click", () => {
    const current =
      root.getAttribute("data-theme") === "light" ? "light" : "dark";
    const next = current === "dark" ? "light" : "dark";
    applyTheme(next);
  });
}

// Mobile menu (burger) toggle
if (menuToggleBtn && navbar) {
  const closeMenu = () => {
    document.body.classList.remove("nav-open");
    menuToggleBtn.classList.remove("is-open");
  };

  menuToggleBtn.addEventListener("click", () => {
    const isOpen = document.body.classList.toggle("nav-open");
    menuToggleBtn.classList.toggle("is-open", isOpen);
  });

  navbar.addEventListener("click", (event) => {
    if (event.target.matches("a")) {
      closeMenu();
    }
  });

  document.addEventListener("click", (event) => {
    if (!document.body.classList.contains("nav-open")) return;

    const clickedToggle = menuToggleBtn.contains(event.target);
    const clickedNavbar = navbar.contains(event.target);

    if (!clickedToggle && !clickedNavbar) {
      closeMenu();
    }
  });

  window.addEventListener("resize", () => {
    if (window.innerWidth > 900) {
      closeMenu();
    }
  });
  document.addEventListener("keydown", (event) => {
    if (event.key === "Escape") {
      closeMenu();
    }
  });
}
